"use client";

import { useState, useEffect } from "react";
import { useParams, useRouter } from "next/navigation";
import { apiGet } from "@/lib/api";
import styles from "./page.module.css";

interface Policy {
  id: number;
  policy_type: string;
  title: string;
  slug: string;
  content: string;
  version?: number;
  language?: string;
  framework?: string;
  published_at?: string;
  is_standard?: boolean;
  files?: PolicyFile[];
}

interface PolicyFile {
  id: number;
  file_name: string;
  file_path: string;
  file_type: string;
  file_size: number;
  download_url?: string;
  created_at: string;
}

export default function PolicyPage() {
  const params = useParams();
  const router = useRouter();
  const slug = params?.slug as string;

  const [isLoading, setIsLoading] = useState(true);
  const [policy, setPolicy] = useState<Policy | null>(null);
  const [error, setError] = useState<string>("");
  const [subdomain, setSubdomain] = useState<string>("");

  useEffect(() => {
    // Get subdomain from hostname
    if (typeof window !== "undefined") {
      const hostname = window.location.hostname;
      const parts = hostname.split(".");
      if (parts.length > 2) {
        // e.g., creativity.localhost:3000 -> creativity
        setSubdomain(parts[0]);
      } else if (parts[0] !== "localhost" && parts[0] !== "127.0.0.1") {
        setSubdomain(parts[0]);
      } else {
        // For localhost, try to get from localStorage or default
        setSubdomain("creativity"); // Default for development
      }
    }
  }, []);

  useEffect(() => {
    if (slug && subdomain) {
      loadPolicy();
    }
  }, [slug, subdomain]);

  const loadPolicy = async () => {
    if (!subdomain || !slug) {
      setError("Subdomain and slug are required");
      setIsLoading(false);
      return;
    }

    setIsLoading(true);
    setError("");

    try {
      // Use public API endpoint (same structure as Dashboard but public)
      const response = await apiGet<Policy>(
        `/companies/policies/${slug}`,
        {},
        subdomain
      );

      if (response.success && response.data) {
        setPolicy(response.data);
      } else {
        setError("Policy not found");
      }
    } catch (err: any) {
      console.error("Error loading policy:", err);
      setError(
        err?.data?.error || err?.data?.message || "Failed to load policy"
      );
    } finally {
      setIsLoading(false);
    }
  };

  const formatFileSize = (bytes: number): string => {
    if (bytes === 0) return "0 Bytes";
    const k = 1024;
    const sizes = ["Bytes", "KB", "MB", "GB"];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round((bytes / Math.pow(k, i)) * 100) / 100 + " " + sizes[i];
  };

  const getPolicyTypeLabel = (policyType: string): string => {
    switch (policyType) {
      case "privacy_policy":
        return "Privacy Policy";
      case "terms_of_service":
        return "Terms of Service";
      default:
        return "Policy";
    }
  };

  const formatDate = (dateString?: string): string => {
    if (!dateString) return "";
    try {
      const date = new Date(dateString);
      return date.toLocaleDateString("en-US", {
        year: "numeric",
        month: "short",
        day: "numeric",
      });
    } catch {
      return dateString;
    }
  };

  const goBack = () => {
    router.push("/");
  };

  return (
    <div className={styles.policyViewContainer}>
      <div className={styles.policyHeader}>
        <button className={styles.btnBack} onClick={goBack}>
          <i className="bi bi-arrow-left"></i>
          <span>Back to Profile</span>
        </button>
      </div>

      <div className={styles.policyContentWrapper}>
        {isLoading && (
          <div className={styles.loadingSpinner}>
            <div className={styles.spinner}></div>
            <p>Loading policy...</p>
          </div>
        )}

        {!isLoading && error && (
          <div className={styles.errorMessage}>
            <div className={styles.errorContent}>
              <i className="bi bi-exclamation-triangle"></i>
              <h2>Policy Not Found</h2>
              <p>{error}</p>
              <button className={styles.btnPrimary} onClick={goBack}>
                Go Back
              </button>
            </div>
          </div>
        )}

        {!isLoading && policy && !error && (
          <div className={styles.policyDetails}>
            <div className={styles.policyHeaderInfo}>
              <div
                className={styles.policyTypeBadge}
                data-type={policy.policy_type}
              >
                {getPolicyTypeLabel(policy.policy_type)}
              </div>
              <h1 className={styles.policyTitle}>{policy.title}</h1>
              <div className={styles.policyMeta}>
                {policy.version && (
                  <span className={styles.metaItem}>
                    <i className="bi bi-version"></i>
                    Version {policy.version}
                  </span>
                )}
                {policy.published_at && (
                  <span className={styles.metaItem}>
                    <i className="bi bi-calendar"></i>
                    Published {formatDate(policy.published_at)}
                  </span>
                )}
                {policy.language && (
                  <span className={styles.metaItem}>
                    <i className="bi bi-globe"></i>
                    {policy.language.toUpperCase()}
                  </span>
                )}
                {policy.framework && (
                  <span className={styles.metaItem}>
                    <i className="bi bi-shield-check"></i>
                    {policy.framework.toUpperCase()}
                  </span>
                )}
              </div>
            </div>

            <div className={styles.policyBody}>
              <div
                className={styles.policyContent}
                dangerouslySetInnerHTML={{ __html: policy.content || "" }}
              />

              {/* Policy Files Section */}
              {policy.files && policy.files.length > 0 && (
                <div className={styles.policyFiles}>
                  <h3>Attachments</h3>
                  <div className={styles.filesList}>
                    {policy.files.map((file) => (
                      <div key={file.id} className={styles.fileItem}>
                        <div className={styles.fileIcon}>
                          <i
                            className={`bi ${
                              file.file_type === "application/pdf"
                                ? "bi-file-earmark-pdf"
                                : file.file_type?.includes("word") ||
                                  file.file_type?.includes("document")
                                ? "bi-file-earmark-word"
                                : "bi-file-earmark"
                            }`}
                          ></i>
                        </div>
                        <div className={styles.fileInfo}>
                          <div className={styles.fileName}>{file.file_name}</div>
                          <div className={styles.fileMeta}>
                            <span>{formatFileSize(file.file_size)}</span>
                            {file.file_type && <span>{file.file_type}</span>}
                          </div>
                        </div>
                        {file.download_url && (
                          <button
                            className={styles.btnDownload}
                            onClick={() => window.open(file.download_url, "_blank")}
                          >
                            <i className="bi bi-download"></i>
                            Download
                          </button>
                        )}
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
          </div>
        )}
      </div>
    </div>
  );
}

